/*
 * Copyright (c) 2019-2022 Actinius
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <zephyr/kernel.h>
#include <zephyr/device.h>
#include <zephyr/devicetree.h>
#include <zephyr/drivers/sensor.h>
#include <stdio.h>

static const struct device *bme_1 = DEVICE_DT_GET(DT_NODELABEL(bme688_1));
static const struct device *bme_2 = DEVICE_DT_GET(DT_NODELABEL(bme688_2));
static const struct device *si = DEVICE_DT_GET(DT_NODELABEL(si7060));
static const struct device *opt = DEVICE_DT_GET(DT_NODELABEL(opt3002));

static bool is_second_bme688_present;

void print_bme688_values(const struct device *dev)
{
    struct sensor_value temp_raw;
    double temp;

    struct sensor_value press_raw;
    double press;

    struct sensor_value humidity_raw;
    double humidity;

    struct sensor_value gas_res_raw;
    double gas_res;

    sensor_sample_fetch(dev);

    sensor_channel_get(dev, SENSOR_CHAN_AMBIENT_TEMP, &temp_raw);
    sensor_channel_get(dev, SENSOR_CHAN_PRESS, &press_raw);
    sensor_channel_get(dev, SENSOR_CHAN_HUMIDITY, &humidity_raw);
    sensor_channel_get(dev, SENSOR_CHAN_GAS_RES, &gas_res_raw);

    temp = sensor_value_to_double(&temp_raw);
    press = sensor_value_to_double(&press_raw);
    humidity = sensor_value_to_double(&humidity_raw);
    gas_res = sensor_value_to_double(&gas_res_raw);

    printf("BME688_1 T: %f; P: %f; H: %f; G: %f\n", temp, press, humidity, gas_res);
}

void print_si7060_value(const struct device *dev)
{
    struct sensor_value temp_raw;
    double temp;

    sensor_sample_fetch(dev);
    sensor_channel_get(dev, SENSOR_CHAN_AMBIENT_TEMP, &temp_raw);
    temp = sensor_value_to_double(&temp_raw);
    printf("SI7060   T: %f\n", temp);
}

void print_opt3002_value(const struct device *dev)
{
    struct sensor_value light_raw;
    double light;

    sensor_sample_fetch(dev);
    sensor_channel_get(dev, SENSOR_CHAN_LIGHT, &light_raw);
    light = sensor_value_to_double(&light_raw);
    printf("OPT3002  LUX: %f\n", light);
}

void main(void)
{
    /* wait for all sensors to become ready */
    bool all_sensors_ready = false;
    while (!all_sensors_ready) {
        all_sensors_ready = true;

        if (!device_is_ready(bme_1)) {
            printk("BME688_1 is not ready\n");
            all_sensors_ready = false;
        }

        if (!device_is_ready(bme_2)) {
            printk("BME688_2 is not ready\n");
            // do not consider the second BME688 missing an error
        } else {
            is_second_bme688_present = true;
        }

        if (!device_is_ready(si)) {
            printk("SI7060 is not ready\n");
            all_sensors_ready = false;
        }

        if (!device_is_ready(opt)) {
            printk("OPT3002 is not ready\n");
            all_sensors_ready = false;
        }

        k_sleep(K_MSEC(500));
    }

    uint32_t sample_count = 0;
    while (true) {
        sample_count++;
        printk("\n\nSample count: %u\n", sample_count);

        print_bme688_values(bme_1);

        if (is_second_bme688_present) {
            print_bme688_values(bme_2);
        }

        print_si7060_value(si);

        print_opt3002_value(opt);

        k_sleep(K_MSEC(3500));
    }
}
